import { z } from 'zod';

// Common schema components
const idSchema = z.number().int().positive();
const slugSchema = z.string().regex(/^[a-z0-9]+(-[a-z0-9]+)*$/, 'Must be lowercase with hyphens');
const urlSchema = z.string().url().nullable();

// Sport schema
export const sportSchema = z.object({
  id: idSchema,
  sportName: slugSchema,
  icon: urlSchema,
  isActive: z.boolean(),
});

// League schema
export const leagueSchema = z.object({
  id: idSchema,
  sportId: idSchema,
  sportName: slugSchema,
  leagueName: slugSchema,
  leagueLogo: urlSchema,
  isActive: z.boolean(),
});

// Participant type enum
export const participantTypeSchema = z.enum(['TEAM', 'DRIVER', 'PLAYER']);

// Participant schema
export const participantSchema = z.object({
  id: idSchema,
  participantType: participantTypeSchema,
  sportId: idSchema,
  sportName: slugSchema,
  leagueIds: z.array(idSchema),
  leagueNames: z.array(slugSchema),
  teamName: slugSchema.nullable(),
  fullName: z.string().nullable(),
  shortName: z.string().nullable(),
  abbreviation: z.string().max(10).nullable(),
  teamLogo: urlSchema,
  isActive: z.boolean(),
});

// WidgetType schema
export const widgetTypeSchema = z.object({
  id: idSchema,
  sportId: idSchema,
  sportName: slugSchema,
  leagueId: idSchema.nullable(),
  leagueName: slugSchema.nullable(),
  widgetName: slugSchema,
  isActive: z.boolean(),
});

// Template schema
export const templateSchema = z.object({
  id: idSchema,
  sportId: idSchema,
  sportName: slugSchema,
  leagueId: idSchema.nullable(),
  leagueName: slugSchema.nullable(),
  participantId: idSchema.nullable(),
  teamName: slugSchema.nullable(),
  widgetTypeId: idSchema,
  widgetName: slugSchema,
  templateName: z.string().min(1),
  darkThemeUrl: urlSchema,
  lightThemeUrl: urlSchema,
  thumbnailDark: urlSchema,
  thumbnailLight: urlSchema,
  featured: z.boolean(),
  isActive: z.boolean(),
});

// Helper to transform optional boolean query params
const optionalBooleanQuery = z.enum(['true', 'false']).optional().transform(v => v === undefined ? undefined : v === 'true');

// Query parameter schemas
export const sportsQuerySchema = z.object({
  isActive: optionalBooleanQuery,
});

export const leaguesQuerySchema = z.object({
  isActive: optionalBooleanQuery,
});

export const participantsQuerySchema = z.object({
  type: participantTypeSchema.optional(),
  isActive: optionalBooleanQuery,
});

export const widgetTypesQuerySchema = z.object({
  sportId: z.string().regex(/^\d+$/).transform(Number).optional(),
  leagueId: z.string().regex(/^\d+$/).transform(Number).optional(),
  isActive: optionalBooleanQuery,
});

export const templatesQuerySchema = z.object({
  sportId: z.string().regex(/^\d+$/).transform(Number).optional(),
  leagueId: z.string().regex(/^\d+$/).transform(Number).optional(),
  widgetTypeId: z.string().regex(/^\d+$/).transform(Number).optional(),
  participantId: z.string().regex(/^\d+$/).transform(Number).optional(),
  featured: optionalBooleanQuery,
  isActive: optionalBooleanQuery,
});

// Path parameter schemas
export const sportIdParamSchema = z.object({
  sportId: z.string().regex(/^\d+$/).transform(Number),
});

export const leagueIdParamSchema = z.object({
  leagueId: z.string().regex(/^\d+$/).transform(Number),
});

// Seed data schema (for import validation)
export const seedDataSchema = z.object({
  version: z.string(),
  sports: z.array(z.object({
    id: idSchema,
    sportName: slugSchema,
    iconUrl: urlSchema,
    isActive: z.boolean().default(true),
  })),
  leagues: z.array(z.object({
    id: idSchema,
    sportId: idSchema,
    sportName: slugSchema,
    leagueName: slugSchema,
    iconUrl: urlSchema,
    isActive: z.boolean(),
  })),
  participants: z.array(z.object({
    id: idSchema,
    participantType: participantTypeSchema,
    sportId: idSchema,
    sportName: slugSchema,
    leagueIds: z.array(idSchema),
    leagueNames: z.array(slugSchema),
    teamName: slugSchema.nullable(),
    fullName: z.string().nullable(),
    shortName: z.string().nullable(),
    abbreviation: z.string().max(10).nullable(),
    teamLogoUrl: urlSchema,
    isActive: z.boolean(),
  })),
  widgetTypes: z.array(widgetTypeSchema),
  templates: z.array(z.object({
    id: idSchema,
    sportId: idSchema,
    sportName: slugSchema,
    leagueId: idSchema.nullable(),
    leagueName: slugSchema.nullable(),
    participantId: idSchema.nullable(),
    teamName: slugSchema.nullable(),
    widgetTypeId: idSchema,
    widgetName: slugSchema,
    templateName: z.string().min(1),
    darkThemeUrl: urlSchema,
    lightThemeUrl: urlSchema,
    thumbnailDarkUrl: urlSchema,
    thumbnailLightUrl: urlSchema,
    featured: z.boolean(),
    isActive: z.boolean(),
  })),
});

// Type exports
export type SportInput = z.infer<typeof sportSchema>;
export type LeagueInput = z.infer<typeof leagueSchema>;
export type ParticipantInput = z.infer<typeof participantSchema>;
export type WidgetTypeInput = z.infer<typeof widgetTypeSchema>;
export type TemplateInput = z.infer<typeof templateSchema>;
export type SeedDataInput = z.infer<typeof seedDataSchema>;
